const genres = {
  fiction: 'Художественная',
  science: 'Научная',
  textbook: 'Учебная',
  children: 'Детская'
};

function initDefaultOrders() {
  const saved = localStorage.getItem('libraryOrders');
  if (!saved) {
    const defaultOrders = {
      active: [
        {
          id: 1,
          title: 'Война и мир',
          genre: 'Художественная',
          returnDate: '2025-03-25',
          status: 'active'
        },
        {
          id: 2,
          title: 'Физика для начинающих',
          genre: 'Научная',
          returnDate: '2025-03-28',
          status: 'active'
        },
        {
          id: 3,
          title: 'История России',
          genre: 'Учебник',
          returnDate: '2025-03-30',
          status: 'active'
        }
      ],
      past: [
        {
          id: 4,
          title: 'Математика для детей',
          genre: 'Детская',
          returnDate: '2025-03-15',
          status: 'completed'
        },
        {
          id: 5,
          title: 'Основы программирования',
          genre: 'Научная',
          returnDate: '2025-03-10',
          status: 'completed'
        },
        {
          id: 6,
          title: 'Приключения Тома Сойера',
          genre: 'Детская',
          returnDate: '2025-03-05',
          status: 'completed'
        }
      ]
    };
    localStorage.setItem('libraryOrders', JSON.stringify(defaultOrders));
  }
}

function getOrders() {
  const saved = localStorage.getItem('libraryOrders');
  return saved ? JSON.parse(saved) : { active: [], past: [] };
}

function saveOrders(orders) {
  localStorage.setItem('libraryOrders', JSON.stringify(orders));
}

function formatDate(dateString) {
  const date = new Date(dateString);
  return date.toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' });
}

function initPage() {
  const currentPage = window.location.pathname.split('/').pop();
  
  if (currentPage === 'index.html' || currentPage === '' || currentPage.includes('index')) {
    initOrdersPage();
  } else if (currentPage === 'book.html' || currentPage.includes('book')) {
    initBookingForm();
  }
}

function initOrdersPage() {
  initDefaultOrders();
  const orders = getOrders();
  displayOrders(orders);
  
  setInterval(() => {
    const updated = getOrders();
    displayOrders(updated);
  }, 500);
}

function displayOrders(orders) {
  const container = document.getElementById('ordersContainer');
  if (!container) return;
  
  if (orders.active.length === 0) {
    container.innerHTML = `
      <div class="empty-message">
        <p>У вас нет активных заказов</p>
      </div>
    `;
    return;
  }
  
  container.innerHTML = orders.active.map(order => `
    <div class="order-item">
      <div class="order-top">
        <h3 class="order-title">${order.title}</h3>
        <span class="genre-badge">${order.genre}</span>
      </div>
      <div class="order-details">
        <div class="detail-row">
          <span>Дата возврата: ${formatDate(order.returnDate)}</span>
        </div>
      </div>
      <button class="extend-btn" onclick="extendOrder(${order.id})">
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <path d="M21.5 2v6h-6M2.5 22v-6h6M2 11.5a10 10 0 0 1 18.8-4.3M22 12.5a10 10 0 0 1-18.8 4.2"></path>
        </svg>
        <span>Продлить на 7 дней</span>
      </button>
    </div>
  `).join('');
}

function extendOrder(orderId) {
  const orders = getOrders();
  const order = orders.active.find(o => o.id === orderId);
  if (order) {
    const currentDate = new Date(order.returnDate);
    currentDate.setDate(currentDate.getDate() + 7);
    order.returnDate = currentDate.toISOString().split('T')[0];
    saveOrders(orders);
    displayOrders(orders);
  }
}

function initBookingForm() {
  initDefaultOrders();
  
  let selectedGenre = null;
  
  const genreOptions = document.querySelectorAll('.genre-option');
  genreOptions.forEach(option => {
    option.addEventListener('click', () => {
      genreOptions.forEach(o => o.classList.remove('selected'));
      option.classList.add('selected');
      selectedGenre = option.dataset.genre;
      updateSubmitButton();
    });
  });
  
  const bookInput = document.getElementById('bookTitle');
  const submitBtn = document.getElementById('submitBtn');
  
  function updateSubmitButton() {
    if (submitBtn) {
      submitBtn.disabled = !(selectedGenre && bookInput.value.trim());
    }
  }
  
  bookInput.addEventListener('input', updateSubmitButton);
  
  const form = document.getElementById('bookingForm');
  if (form) {
    form.addEventListener('submit', (e) => {
      e.preventDefault();
      
      if (selectedGenre && bookInput.value.trim()) {
        const orders = getOrders();
        const genreName = genres[selectedGenre];
        
        const returnDate = new Date();
        returnDate.setDate(returnDate.getDate() + 14);
        
        const newOrder = {
          id: Date.now(),
          title: bookInput.value.trim(),
          genre: genreName,
          returnDate: returnDate.toISOString().split('T')[0],
          status: 'active'
        };
        
        orders.active.push(newOrder);
        saveOrders(orders);
        
        window.location.href = 'index.html';
      }
    });
  }
}

if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initPage);
} else {
  initPage();
}

